require "TimedActions/ISBaseTimedAction"

RecycleVehicleAction = ISBaseTimedAction:derive("RecycleVehicleAction")

function RecycleVehicleAction.getBaseArea(vehicle)
    local script = vehicle:getScript()
    local vModifier = script:getMechanicType() == 2 and 1 or -2

    local extX = script:getExtents():x()
    local extY = script:getExtents():y()
    local extZ = script:getExtents():z()

    return math.ceil(2 * (extX + extZ) * extY + extX * extZ) + vModifier
end

function RecycleVehicleAction:isValid()
    return self.vehicle and not self.vehicle:isRemovedFromWorld();
end

function RecycleVehicleAction:waitToStart()
    self.character:faceThisObject(self.vehicle)
    return self.character:shouldBeTurning()
end

function RecycleVehicleAction:update()
    self.character:faceThisObject(self.vehicle)
    self.character:setMetabolicTarget(Metabolics.HeavyWork)

    if not self.character:getEmitter():isPlaying(self.sound) then
        self.sound = self.character:playSound("BlowTorch")
    end
end

function RecycleVehicleAction:start()
    self.item = self.character:getPrimaryHandItem()
    self:setActionAnim("BlowTorch")
    self:setOverrideHandModels(self.item, nil)
    self.sound = self.character:playSound("BlowTorch")
end

function RecycleVehicleAction:stop()
    if self.sound ~= 0 then
        self.character:getEmitter():stopSound(self.sound)
    end
    ISBaseTimedAction.stop(self)
end

function RecycleVehicleAction:perform()
    if self.sound ~= 0 then
        self.character:getEmitter():stopSound(self.sound)
    end

    -- Vehicle
    local vehicleName = self.vehicle:getScript():getName()
    local vehicleType = self.vehicle:getScript():getMechanicType()

    -- Basic outcome
    local metalSheet = math.floor(self.vArea / 7)
    local metalSheetSmall = math.floor((self.vArea - metalSheet * 4) / 3)
    local metalBar = math.floor(self.vArea - metalSheet * 4 - metalSheetSmall * 2)
    -- Other elements
    local metalPipe = 0
    local metalScrap = 0
    local metalUnusable = ZombRand(3)
    local electronicsScrap = 1
    local electricWire = ZombRand(3)
    local fabric = 0
    local lightsItems = {}
    local tireItems = {}
    -- Items left in the vehicle
    local junk = {}

    -- Special vehicles and trailers
    if vehicleName == "VanRadio" or vehicleName == "VanAmbulance" then
        electronicsScrap = electronicsScrap + 3
        electricWire = electricWire + 3
    elseif vehicleName:find("Trailer") then
        electronicsScrap = 0
        electricWire = 1
    end

    for i = 1, self.vehicle:getPartCount() do
        local part = self.vehicle:getPartByIndex(i - 1)
        local partId = part:getId()
        local partCondition = part:getCondition()

        local partItem = part:getInventoryItem()
        if partItem then
            -- The part is installed
            local partItemCondition = partItem:getCondition()
            local partItemType = partItem:getType()

            -- Collect items from all containers
            if part:isContainer() and not part:getContainerContentType() then
                local container = part:getItemContainer()
                for j = 1, container:getItems():size() do
                    table.insert(junk, container:getItems():get(j - 1))
                end
            end

            -- Don't give anything useful for brakes
            if partId:find("Brake") then
                metalUnusable = metalUnusable + 1
            end

            -- Give something for other parts
            -- TrunkDoorWreck is a part from Crashed Cars Mod (Build 41)
            if partItemCondition > 30 and partId ~= "TrunkDoorWreck" then
                if partId:find("Door") then
                    if partItemCondition > 70 then
                        metalSheet = metalSheet + 1
                    else
                        metalSheetSmall = metalSheetSmall + 1
                    end
                elseif partId:find("GasTank") then
                    if partItemCondition > 70 then
                        if partItemType:find("^Small") then
                            metalSheetSmall = metalSheetSmall + 2
                        elseif partItemType:find("^Normal") then
                            metalSheet = metalSheet + 1
                        elseif partItemType:find("^Big") then
                            metalSheet = metalSheet + 2
                        end
                    else
                        metalSheetSmall = metalSheetSmall + 1
                    end
                elseif partId:find("Muffler") then
                    if partItemCondition > 70 then
                        metalSheetSmall = metalSheetSmall + 1
                        metalPipe = metalPipe + 2
                    else
                        metalPipe = metalPipe + 1
                    end
                elseif partId:find("Suspension") then
                    metalPipe = metalPipe + 1
                elseif partId:find("Tire") then
                    table.insert(tireItems, partItem)
                elseif partId:find("Headlight") then
                    table.insert(lightsItems, partItem)
                elseif partId:find("Radio") then
                    electronicsScrap = electronicsScrap + 1
                    electricWire = electricWire + 1
                elseif partId:find("Seat") then
                    if partItemType:find("^Sports") then
                        fabric = fabric + 1
                    elseif partItemType:find("^Normal") then
                        fabric = fabric + 2
                    elseif partItemType:find("^Big") then
                        fabric = fabric + 3
                    end
                end
            else
                -- The part item's condition is too bad to give anything useful
                if partId:find("Door") or partId:find("GasTank") or partId:find("Muffler") or partId:find("Suspension") then
                    metalUnusable = metalUnusable + 1
                end
            end
        else
            -- The part is not replaceable or part item is uninstalled
            if partId:find("lightbar") then
                electronicsScrap = electronicsScrap + 1
                -- Create light bulbs as they don't exist
                for _ = 1, 2 do
                    local item = InventoryItemFactory.CreateItem("Base.LightBulb")
                    item:setCondition(partCondition)
                end
            end
        end
    end

    -- Drop junk
    for i = 1, #junk do
        self:dropItem(junk[i])
    end
    -- Create items and drop them on the ground
    local success = 20 + self.weldingLvl * 7
    -- Metal
    for _ = 1, metalSheet do
        if ZombRand(0, 100) < success then
            self:dropItem("Base.SheetMetal")
        else
            metalSheetSmall = metalSheetSmall + 1
        end
    end
    for _ = 1, metalSheetSmall do
        if ZombRand(0, 100) < success then
            self:dropItem("Base.SmallSheetMetal")
        else
            metalScrap = metalScrap + 1
        end
    end
    for _ = 1, metalBar do
        if ZombRand(0, 100) < success then
            self:dropItem("Base.MetalBar")
        else
            metalScrap = metalScrap + 1
        end
    end
    for _ = 1, metalPipe do
        if ZombRand(0, 100) < success then
            self:dropItem("Base.MetalPipe")
        else
            metalScrap = metalScrap + 1
        end
    end
    -- Do after all metal items
    for _ = 1, metalScrap do
        if ZombRand(0, 100) < success + 10 then
            self:dropItem("Base.ScrapMetal")
        end
    end
    for _ = 1, metalUnusable do
        self:dropItem("Base.UnusableMetal")
    end

    -- Tires
    for i = 1, #tireItems do
        if ZombRand(0, 100) < success then
            local tire = self:dropItem(tireItems[i])
            tire:setCondition(ZombRand(10, tire:getCondition()))
        end
    end

    -- Electrics
    for _ = 1, electricWire do
        if ZombRand(0, 100) < success then
            self:dropItem("Radio.ElectricWire")
        end
    end
    for _ = 1, electronicsScrap do
        if ZombRand(0, 100) < success then
            self:dropItem("Base.ElectronicsScrap")
        end
    end
    for i = 1, #lightsItems do
        if ZombRand(0, 100) < success then
            local lightBulb = self:dropItem(lightsItems[i])
            lightBulb:setCondition(ZombRand(10, lightBulb:getCondition()))
        end
    end

    -- Fabric
    for _ = 1, fabric do
        if ZombRand(0, 100) < success then
            -- Leather for sport cars, ripped sheets for the other types
            if vehicleType == 3 then
                self:dropItem("Base.LeatherStrips")
            else
                self:dropItem("Base.RippedSheets")
            end
        end
    end

    -- Drop key from the player's inventory if exists
    local key = self.character:getInventory():haveThisKeyId(self.vehicle:getKeyId())
    if key then
        --key:getContainer():Remove(key)
        self:dropItem(key)
    end

    -- Use BlowTorch
    for _ = 1, self.propaneNeeded do
        self.item:Use()
    end

    -- Award with XP
    self.character:getXp():AddXP(Perks.MetalWelding, 5 + 2 * (metalSheet + metalSheetSmall + metalBar + metalPipe + metalScrap))
    self.character:getXp():AddXP(Perks.Electricity, 2 * (electricWire + electronicsScrap + #lightsItems))

    -- Remove the vehicle
    if isClient() then
        sendClientCommand(self.character, "vehicle", "remove", { vehicle = self.vehicle:getId() })
    else
        self.vehicle:permanentlyRemove()
    end

    -- Remove the Timed Action from stack
    ISBaseTimedAction.perform(self);
end

function RecycleVehicleAction:dropItem(item)
    return self.vehicle:getSquare():AddWorldInventoryItem(item, ZombRandFloat(0, 0.9), ZombRandFloat(0, 0.9), 0.0)
end

function RecycleVehicleAction:new(character, vehicle, propaneNeeded)
    local o = {}
    setmetatable(o, self)
    self.__index = self
    o.stopOnWalk = true
    o.stopOnRun = true
    o.character = character
    o.vehicle = vehicle
    o.propaneNeeded = propaneNeeded
    o.weldingLvl = character:getPerkLevel(Perks.MetalWelding)
    o.vArea = RecycleVehicleAction.getBaseArea(vehicle)
    o.maxTime = o.vArea * 60 - o.weldingLvl * o.vArea * 1.2

    if ISVehicleMechanics.cheat then
        o.maxTime = 1
    end
    return o
end